/////////////////////////////////////////////////////////////////
//                                                             //
//  FF1 VME interface debugging tool                           //
//                                                             //
//  Author: Matt Noy                                           //
//  date: 21/03/2003                                           //
//  notes:                                                     //
//                                                             //
//                                                             //
/////////////////////////////////////////////////////////////////

// added fed specific methods  by jac starting 06.05.2003


#ifndef _FFv1Object_hh_
#define _FFv1Object_hh_


//#include "StopWatch.hh"
//#include "VMEDummyBusAdapter.hh"

#include "VMEBusAdapterInterface.hh"
#ifdef BUILD_NIVXI
#include "MXI2x86LinuxBusAdapter_test.hh" 
#else
#include "SBS620x86LinuxBusAdapter.hh" 
#endif

#include "VMEAddressTableASCIIReader.hh"
#include "VMEDevice.hh"
#include "VMEAddressTable.hh"
#include <sstream>

#include <unistd.h>

#include <vector>

#include <stdio.h>

// simple FED MMAP byte offsets
const unsigned int FED_Serial_BRAM = 0x0000;

const unsigned int FED_Serial_Write = 0x0800;
const unsigned int FED_Serial_Read = 0x0804;
const unsigned int FED_Serial_Status = 0x080c;

const unsigned int FED_Readout_Buffer_Length = 0x0820;
const unsigned int FED_Readout_Event_Ctr = 0x0824;
const unsigned int FED_Readout_Event_Length = 0x0828;
const unsigned int FED_Readout_CSR = 0x082c;

const unsigned int FED_Firmware_ID = 0x0830;
const unsigned int FED_Clock_Select = 0x0834;
const unsigned int FED_Reset = 0x0838;
const unsigned int FED_VME_Status = 0x083c;

const unsigned int FED_TTC_Clk_Ctr = 0x0840;
const unsigned int FED_BP_Clk_Ctr = 0x0844;

const unsigned int FED_Readout_BRAM = 0x8000;

const unsigned int MAX_BUFFER_LENGTH = 1024*4;
unsigned long local_buffer[MAX_BUFFER_LENGTH]; // test destination for vme event readout



class FFv1Object// ,Task
{
public:
  FFv1Object(unsigned long baseAddr, string addrTable);
  ~FFv1Object();
  char userPrompt();
  int FFv1Object::preliminaryTest(string & addrItem);  

  int FFv1Object::checkData(unsigned long * dataIn, unsigned long * dataOut, unsigned long len);
  int FFv1Object::readBlockMemTest();
  int FFv1Object::writeBlockMemTest();

  int FFv1Object::ffv1BlockWrite(unsigned long * src, unsigned long length, unsigned long offset=0);
  int FFv1Object::ffv1BlockRead(unsigned long * dest, unsigned long length, unsigned long offset=0);
  int FFv1Object::ffv1SingleWrite(unsigned long src, unsigned long offset=0); // passes data by value
  int FFv1Object::ffv1SingleRead(unsigned long * dest, unsigned long offset=0);

  int SetOptoRx(int chip, int value);
  int SetRunMode(int chip, int mode);
  int SetScopeLength(int chip, int len);
  int SendSWTrig(void);

  int ReadOptoRx(int chip, unsigned int* data);
  int ReadScopeLength(int chip, unsigned int* data);

  int EnableAPVChans(int chip, int data);
  int SetFrameThresholds(int chip, int data);

  int SetLM82(int chip, int value);
  int ReadLM82(int chip, unsigned int* data);
  int ReadTemp(int chip, int* data);

  int SetDAC(int chip, int chan, int value);
  int SetAD8802(int chip, int value);
  int ReadAD8802(int chip, unsigned int* data);

  unsigned int fedCmd(int chip_id, int designator, int cmd_len, int r_w);
  unsigned int cdcCmd(int chip_id, int designator, int cmd_len);

  int FillVMEMem(int data);
  int FillVMEEventMem(int data);
  int DumpVMEMem(void);

  int SetClockSkew(int fe_chip, int delay_chip, int delay_chan, int coarse, int fine);

  int BECommand(int be_code, int be_rw, unsigned int* be_data, int verbose);
  int DisplayBEStatus(int flag);
  int InitBE(int run_type, unsigned int trigger_source);

  int TestFrameFinding(int num_triggers, int num_microsecs);

  int FECommand(int fe_id, int fe_code, int fe_rw, unsigned int* fe_data, int verbose);

  int TestReadout(int verbose);
  int ClearReadoutCSR(int verbose);
  int CheckEvent(unsigned long* event_buffer, unsigned long event_length);
  int DisplayVMEStatus(int verbose);

  int TestTTCrx(int* data);
  int SetTTCrx(int value);
  int ReadTTCrx(unsigned int* data);

  int ResetFED(int verbose);

  int SelectClock(int clock, int verbose);
  int DisplayClockCounters(int verbose);

private:
  void FFv1Object::wait(unsigned long ws, unsigned long wms);
  
protected:

  VMEAddressTableASCIIReader ffv1AddrTblRdr_;
  VMEAddressTable ffv1AddrTable_;

#ifdef BUILD_NIVXI
  MXI2x86LinuxBusAdapter_test ffv1BusAdapter_;
#else
  SBS620x86LinuxBusAdapter ffv1BusAdapter_;
#endif
  VMEDevice ffv1Device_;

};

//
//constructor with non default base address and address table...
//

FFv1Object::FFv1Object(unsigned long baseAddr, string addrTable):
  ffv1AddrTblRdr_(addrTable),
  ffv1AddrTable_("I2C VME Address Map",ffv1AddrTblRdr_),
  ffv1BusAdapter_(0),
  ffv1Device_(ffv1AddrTable_,ffv1BusAdapter_,baseAddr)
{
}
FFv1Object::~FFv1Object()
{
}
// jac changed to take long rather than *long
int FFv1Object::ffv1SingleWrite(unsigned long src, unsigned long offset=0)
{
   try
    {
      ffv1Device_.unmaskedWrite("FFv1BaseItemSingleDataNonPriv",src,HAL_NO_VERIFY,(offset<<2));
    }
   catch(HardwareAccessException & e)
    {
      cout<<"exception from ffv1SingleWrite(), unmaskedWrite() :\n "<<e.what()<<endl;
      return -1;
    }
  return 0;
}

int FFv1Object::ffv1SingleRead(unsigned long * dest,  unsigned long offset=0)
{
   try
    {
      ffv1Device_.unmaskedRead("FFv1BaseItemSingleDataNonPriv",dest,(offset<<2));
    }
   catch(HardwareAccessException & e)
    {
      cout<<"exception from ffv1SingleRead(), unmaskedRead() :\n "<<e.what()<<endl;
      return -1;
    }
  return 0;
}

int FFv1Object::ffv1BlockWrite(unsigned long * src,  unsigned long length, unsigned long offset=0)
{
   try
    {
      ffv1Device_.writeBlock("FFv1BaseItemBlockNonPriv",length, (char*)src,HAL_NO_VERIFY,HAL_DO_INCREMENT,offset);
    }
   catch(HardwareAccessException & e)
    {
      cout<<"exception from ffv1BlockWrite(), writeBlock() :\n "<<e.what()<<endl;
      return -1;
    }
  return 0;
}
int FFv1Object::ffv1BlockRead(unsigned long * dest, unsigned long length, unsigned long offset=0)
{
  try
    {
      ffv1Device_.readBlock("FFv1BaseItemBlockNonPriv",length, (char*)dest, HAL_DO_INCREMENT,offset);
    }
  catch(HardwareAccessException & e)
    {
      cout<<"exception from ffv1BlockRead(), readBlock() :\n "<<e.what()<<endl;
      return -1;
    }
  return 0;
} 
int FFv1Object::checkData(unsigned long * dataIn, unsigned long * dataOut, unsigned long len)
{
  int errors=0;
  for(unsigned long i=0; i<len; i++)
    {
      if(dataIn[i]!=dataOut[i])
	{
	  cout<<"data mismatch: "<<hex<<dataIn[i]<<" not equal to "<<hex<<dataOut[i]<<" at index "<<dec<<i<<endl;
	  errors++;
	}
    }
  
  return errors;
}


int FFv1Object::readBlockMemTest()
{
 //start of the block read test.  
  cout<<"performing block read memory test"<<endl;
  const unsigned long memSize=0x40;
  unsigned long dataIn[memSize]={0};  
  unsigned long dataOut[memSize]={0};
  unsigned long i=0;
  for(i=0; i<memSize;i++)
    {
      dataOut[i]=i;
    }
  try
    {
      for(i=0; i<memSize; i++)
	{
	  ffv1Device_.unmaskedWrite("FFv1BaseItemSingleDataNonPriv",dataOut[i],HAL_DO_VERIFY,(i<<2));
    	}
    }
  catch(HardwareAccessException & e)
    {
      cout<<"exception from readBlockMemTest, unmaskedWrite :\n "<<e.what()<<endl;
      return -1;
    }
  cout<<"single write data complete, reading back..."<<endl;
  try
    {
      ffv1Device_.readBlock("FFv1BaseItemBlockNonPriv",memSize, (char*)dataIn, HAL_DO_INCREMENT,0x0);
      //for(i=0; i<memSize; i++)//for testing only
      //	ffv1Device_.unmaskedRead("FFv1BaseItemSingleDataNonPriv",&dataIn[i],(i<<2));
    }
  catch(HardwareAccessException & e)
     {
      cout<<"exception from readBlockMemTest, readBlock :\n "<<e.what()<<endl;
      return -1;
    }
  
  cout<<"block read back complete, checking..."<<endl;
  int err=0;
  if((err=checkData(dataIn,dataOut,memSize)))
    {
      cout<<"there were "<<err<<" errors in the transfer: unmaskedWrite ->readBlock"<<endl;
      return err;
    }
  //end of the block read test.
  cout<<"completed\n";

  return 0;
}


int FFv1Object::writeBlockMemTest()
{
  cout<<"performing block write memory test"<<endl;
  const unsigned long memSize=0x40;
  unsigned long dataOut[memSize]={0};
  unsigned long dataIn[memSize]={0};
  unsigned long i=0;
  for(i=0; i<memSize;i++)
    dataOut[i]=rand();
  
  //beginning of the block write test.

  cout<<"making single write..."<<endl;
  try
    {
      ffv1Device_.writeBlock("FFv1BaseItemBlockNonPriv",memSize,(char*)dataOut,HAL_NO_VERIFY,HAL_DO_INCREMENT,0x0);
    }
  catch(HardwareAccessException & e)
    {
      cout<<"exception from rndBlockMemTest(), writeBlock() :\n "<<e.what()<<endl;
      return -1;
    }
  cout<<"block write complete, preparing to read back"<<endl;
  try
    {
      for(i=0; i<memSize; i++)
	ffv1Device_.unmaskedRead("FFv1BaseItemSingleDataNonPriv",&dataIn[i],(i<<2));
    }
  catch(HardwareAccessException & e)
    {
      cout<<"exception from rndBlockMemTest(), unmaskedWrite() :\n "<<e.what()<<endl;
      return -1;
    }
  cout<<"single read back complete, checking..."<<endl;
  int err=0;
  if((err=checkData(&dataIn[0],&dataOut[0],memSize)))
    {
      cout<<"there were errors "<<err<<" in the transfer: writeBlock -> unmaskedRead "<<endl;
      return err;
    }
  
  //end of the block write test, 
  return 0;
}

int FFv1Object::preliminaryTest(string & addrItem)
{

  unsigned long tempIn=0;
  unsigned long tempOut=0;

  try
  {
      
    tempOut=0xaaaaaaaa;
    ffv1Device_.unmaskedWrite(addrItem,tempOut,HAL_NO_VERIFY,0);
    ffv1Device_.unmaskedRead(addrItem,&tempIn,0);  
    if(tempIn!=tempOut)
      cout<<"error in reading back"<<endl;
    
    tempOut=0x55555555;
    ffv1Device_.unmaskedWrite(addrItem,tempOut,HAL_NO_VERIFY,0);
    ffv1Device_.unmaskedRead(addrItem,&tempIn,0);  
    if(tempIn!=tempOut)
      cout<<"error in reading back"<<endl;  
    
    tempOut=0xffffffff;
    ffv1Device_.unmaskedWrite(addrItem,tempOut,HAL_NO_VERIFY,0);
    ffv1Device_.unmaskedRead(addrItem,&tempIn,0);  
    if(tempIn!=tempOut)
      cout<<"error in reading back"<<endl;
    
    tempOut=0xfaceface;
    ffv1Device_.unmaskedWrite(addrItem,tempOut,HAL_NO_VERIFY,0);
    ffv1Device_.unmaskedRead(addrItem,&tempIn,0);  
    if(tempIn!=tempOut)
      cout<<"error in reading back"<<endl;
  
    tempOut=0x0;
    ffv1Device_.unmaskedWrite(addrItem,tempOut,HAL_NO_VERIFY,0);
    ffv1Device_.unmaskedRead(addrItem,&tempIn,0);  
    if(tempIn!=tempOut)
      cout<<"error in reading back"<<endl;   

  }
  catch(BusAdapterException & e) 
    {
      cout<<"mn: exception in function preliminaryTest() :\n "<<e.what()<<endl;
      return 1;
    }
  return 0;
}






char FFv1Object::userPrompt()
{
  char choice=-1;
  string temp;
  bool isValid=true;
  do
    {
//        cout << endl
//  	  <<"############################################\n"
//  	  <<"#------------------------------------------#\n"
//  	  <<"#--------- FED Operation Menu -------------#\n"
//  	  <<"#------------------------------------------#\n"
//  	  <<"############################################\n"
//  	  <<"# Please enter a choice:                   #\n"
//  	  <<"#                                          #\n"
//  	  <<"# a set optorx controls                    #\n"
//  	  <<"# b set scope mode                         #\n"
//  	  <<"# c set clock skews                        #\n"
//  	  <<"# d set scope mode length                  #\n"
//  	  <<"# e set frame finding mode                 #\n"
//  	  <<"# f set frame thresholds                   #\n"
//  	  <<"# g enable frame finding on apv channels   #\n"
//      	  <<"#                                          #\n"
//  	  <<"# h BE FPGA commands                       #\n"
//      	  <<"#                                          #\n"
//   	  <<"# i initialise frame finding (do it all)   #\n"
//      	  <<"#                                          #\n"
//  	  <<"# k set lm82 lines                         #\n"
//  	  <<"# l read lm82 lines                        #\n"
//  	  <<"#                                          #\n"
//  	  <<"# m read optorx settings                   #\n"
//  	  <<"# n read scope length                      #\n"
//  	  <<"# o lm82   (not ready)                     #\n"
//  	  <<"# p ad8802 set dac offsets                 #\n"
//  	  <<"# q ad8802 line status                     #\n"
//  	  <<"#                                          #\n"
//  	  <<"# r show VME data buffer (not ready)       #\n"
//  	  <<"# s fill VME data buffer (not ready)       #\n"
//       	  <<"#                                          #\n"
//  	  <<"# t send software trigger                  #\n"
//       	  <<"#                                          #\n"
//  	  <<"# u show VME command buffer                #\n"
//  	  <<"# v fill VME command buffer                #\n"
//  	  <<"# w set fe fpga (1-8 ; 15 for Broadcast)   #\n"
//  	  <<"# x exit                                   #\n"
//  	  <<"#                                          #\n"
//  	  <<"############################################"<<endl;
      
      cout << endl
	  <<"###############################################################################\n"
	  <<"# Please enter a choice:                                                      #\n"
	  <<"#                                                                             #\n"
	  <<"# a) set optorx, b) scope mode, c) set skews, d) scope len                    #\n"
	  <<"# e) set frame mode, f) thresh, g) enable apvs                                #\n"
    	  <<"#                                                                             #\n"
	  <<"# h) BE FPGA commands, i) init FED for run, j) show BE regs                   #\n"
    	  <<"# k) Test Frame Finding                                                       #\n"
	  <<"# m) rd optorx, n) rd scope len, o) FE commands                               #\n"
	  <<"#                                                                             #\n"
	  <<"# t) s/w trigger & clear evt csr                                              #\n"
	  <<"# u) show command buffer v) fill command buffer                               #\n"
	  <<"# w) set fe fpga (1-8 ; 15 for Broadcast), x) exit                            #\n"
	  <<"# y) show VME status, z) test readout                                         #\n"
	  <<"#                                                                             #\n"
	  <<"# 1) FED Reset, 2) Select Clock(resets FED if successful!)                    #\n"
	  <<"# 3) s/w trigger only, 4) clear evt csr, 6) Clock ctrs                        #\n"
	  <<"###############################################################################"<<endl;

      cin>>temp;
      choice=temp[0];
      temp.erase();
      choice=tolower(choice);
      if((choice>='a' && choice<='z') || (choice>='0' && choice<='9'))
	isValid=false;
      else
	cout<<"\n*********** unrecognised choice, ***********\n*********** please enter another ***********\n\n"
	    <<endl;
    }while(isValid);

  return choice;
}

/*--------------------------wait----------------------------------*/
/*wait takes seconds and milliseconds to wait for.
 *the ws variable needn't be used for delays
 *of more that a sec, but must be included in the
 *argument list. 
 *
 *matt noy, 05/2002
 */

//modified by m.noy on 11-02-2003 to remove the printf()
//in order to use with c++ under rh linux 7.2

//modified by m.noy 12-02-2003 to use microseconds instead
//and the arguments changed to type unsigned long 
void FFv1Object::wait(unsigned long ws, unsigned long  wus)
{
	struct timespec rqtp;/*requested time interval*/
	struct timespec rmtp;/*remaining time interval*/
	int nsret=-1;/*nanosleep return value*/
	if(wus>1000000)/*ensure that the correct range is used for millisec */
	{
		ws=static_cast<unsigned long>(wus/1000000);
		wus=wus%1000000;
	}

	//printf("WAIT: Seconds = %d, Milliseconds = %d.\n", ws, wms);

	rqtp.tv_sec=ws;/*set the time structure*/
	rqtp.tv_nsec=1000*wus;

	if((nsret=nanosleep(&rqtp, &rmtp))!=0)
	{
		rqtp=rmtp;
		cout<<"WAIT: error in nanosleep, nsret = %"<<nsret<<endl;
		return;
	}
}

int FFv1Object::SetOptoRx(int chip, int data)
{
  cout << "FE Chip # " << chip << " OptoRx Controls = $ " << hex << data << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x16, 8, 0);
  unsigned int fed_data = data<<(32-8);
  cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';

  // return 0;

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data, 0x1);

  ffv1SingleWrite(2, 0x200);  // send cmd string

  return 0;
}


int FFv1Object::SetScopeLength(int chip, int len)
{
  cout << dec << "FE Chip # " << chip << " Scope Length = " << len << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x11, 10, 0);
  unsigned int fed_data = len<<(32-10);
  cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';

  //  return 0;

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data, 0x1);

  ffv1SingleWrite(2, 0x200);  // send cmd string

  return 0;
}

int FFv1Object::SetRunMode(int chip, int mode)
{
  unsigned int mode_bits;

  switch (mode) {
  case 1 :
    cout << dec << "FE Chip # " << chip << " Run Mode  = Scope Mode" << '\n';
    mode_bits = 0x05;
    break;
  case 2 :
    cout << dec << "FE Chip # " << chip << " Run Mode  = Frame Finding/VIRGIN RAW Data" << '\n';
    mode_bits = 0x06;
    break;
  case 3 :
    cout << dec << "FE Chip # " << chip << " Run Mode  = Frame Finding/ZERO SUPPRESSED Data" << '\n';
    mode_bits = 0x0A;
    break;
  default :
    cout << " Sorry Illegal Run Mode" << '\n';
    return 1;
    break;
  }


  unsigned int fed_cmd = fedCmd(chip, 0x10, 5, 0);
  unsigned int fed_data = mode_bits<<(32-5);
  cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';

  //  return 0;

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data, 0x1);

  ffv1SingleWrite(2, 0x200);  // send cmd string

  return 0;
}

int FFv1Object::EnableAPVChans(int chip, int data)
{
   data = 0xffffff; // enable all 24 channels
  //data = 0x000000; // disable some
  //cout << "Test disabling APV chans" << '\n';


  cout << "FE Chip # " << chip << " Enabling Frame Finding on ALL APV Chans (1-24)" << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x02, 24, 0);
  unsigned int fed_data = data<<(32-24);
  cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';

  // return 0;

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data, 0x1);

  ffv1SingleWrite(2, 0x200);  // send cmd string

  return 0;
}

int FFv1Object::SetFrameThresholds(int chip, int data)
{
  cout << "FE Chip # " << chip << " Frame Threshold for ALL 12 fibres = " << data << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x04, 60, 0);

  unsigned int fed_data1 = data << 27 | data << 22 | data << 17 | data << 12 | data << 7 | data << 2 | data >> 3;
  unsigned int fed_data2 = data << 29 | data << 24 | data << 19 | data << 14 | data << 9 | data << 4;
  cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data1 = $ " << fed_data1 << " ; fed_data2 = $ " << fed_data2 << '\n';

  // return 0;

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data1, 0x1); 
  ffv1SingleWrite(fed_data2, 0x2);

  ffv1SingleWrite(3, 0x200);  // send cmd string

  return 0;
}

int FFv1Object::SendSWTrig(void)
{
  cout << " Send SW Trigger" << '\n';

  unsigned int fed_cmd = fedCmd(9, 0x0f, 1, 0);  // any cmd to be fpga generates s/w pulse
  unsigned int fed_data = 0x1<<(32-1);
  cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';

  // return 0;

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data, 0x1); // data is irrelevant

  ffv1SingleWrite(2, 0x200);  // send cmd string

  return 0;
}

int FFv1Object::ReadOptoRx(int chip, unsigned int* data)
{
  cout << "Readback FE Chip # " << chip << " OptoRx Controls" << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x16, 8, 1);
  cout << "fed_cmd = $ " << hex << fed_cmd << '\n';

  // return 0;

  ffv1SingleWrite(fed_cmd, 0x201);  // sends read cmd
  usleep(100);
  unsigned long fed_data = *data;
  ffv1SingleRead(&fed_data, 0x0);  // readback memory
  *data = fed_data >> (32-8);

  return 0;
}

int FFv1Object::ReadScopeLength(int chip, unsigned int* data)
{
  cout << "Readback FE Chip # " << chip << " Scope Length" << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x11, 10, 1);
  cout << "fed_cmd = $ " << hex << fed_cmd << '\n';

  // return 0;

  ffv1SingleWrite(fed_cmd, 0x201);  // sends read cmd
  usleep(100);
  unsigned long fed_data = *data;
  ffv1SingleRead(&fed_data, 0x0);  // readback memory
  *data = fed_data >> (32-10);

  return 0;
}

int FFv1Object::ReadTemp(int chip, int* data)
{
  cout << "Read FE Chip # " << chip << " Local Temp" << '\n';
  // assumes command register set to Local Temp

  // send 7 fixed address bits
  int serial_data = 0x18;
  // int serial_data = 0x4e;
  unsigned int lm82_ctrl = 0; // sdi/sdo/sck
  unsigned int temp = 0;
  unsigned int lm82_ack = 0xff;
  unsigned int d7, d6, d5, d4, d3, d2, d1, d0;
  unsigned int readbit = 0xff;

    ReadLM82(chip, &lm82_ctrl);
    cout << " start lm82 data = " << lm82_ctrl << '\n';

 // Start : clock high, data high -> low
    //    SetLM82(chip, 0x3);
    // usleep(10);
    //    SetLM82(chip, 0x1);

//      for (int i=0; i<10; i++)
//        cout << "waiting..." << '\n';
    //    usleep(100);

    // ReadLM82(chip, &lm82_ctrl);
    ///cout << " lm82 data = " << hex << lm82_ctrl << '\n';

    SetLM82(chip, 0x3);
    //SetLM82(chip, 0x3);
    //SetLM82(chip, 0x3);
    //SetLM82(chip, 0x1);
    //SetLM82(chip, 0x1);
    SetLM82(chip, 0x1);
    SetLM82(chip, 0x0);
    SetLM82(chip, 0x1);
    //SetLM82(chip, 0x0);
    SetLM82(chip, 0x0);
    SetLM82(chip, 0x1);
    //SetLM82(chip, 0x0);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    //SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    // SetLM82(chip, 0x2);
    SetLM82(chip, 0x0);
    SetLM82(chip, 0x1);
    // SetLM82(chip, 0x0);
    //
    SetLM82(chip, 0x0);
    SetLM82(chip, 0x1);
    //SetLM82(chip, 0x0);
    //SetLM82(chip, 0x2);
    //SetLM82(chip, 0x3);
    //SetLM82(chip, 0x2);
 //
    SetLM82(chip, 0x0);
    SetLM82(chip, 0x1);
   
    // SetLM82(chip, 0x0);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    // usleep(50);
    ReadLM82(chip, &readbit);
   
    //SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);

    // usleep(100);

    ReadLM82(chip, &lm82_ack);
    // cout << " ack lm82 data = " << lm82_ack << '\n';

    //    SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    ReadLM82(chip, &d7);

    //    SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    ReadLM82(chip, &d6);

    //    SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    ReadLM82(chip, &d5);

    //    SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    ReadLM82(chip, &d4);

    //    SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    ReadLM82(chip, &d3);

    //    SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    ReadLM82(chip, &d2);

    // SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    ReadLM82(chip, &d1);

    // SetLM82(chip, 0x2);
    SetLM82(chip, 0x2);
    SetLM82(chip, 0x3);
    ReadLM82(chip, &d0);


    //ReadLM82(chip, &lm82_ctrl);
    //cout << " ack lm82 data = " << lm82_ctrl << '\n';

    cout << " readbit  = " << readbit << '\n';
    cout << " ack = " << lm82_ack << '\n';
    cout << " d7-0 = " << d7 << " ; " << d6 << " ; " << d5 << " ; " << d4 << " ; " << d3 << " ; " << d2 << " ; " << d1 << " ; " << d0 << '\n';

    ReadLM82(chip, &lm82_ctrl);
    cout << " ack lm82 data = " << lm82_ctrl << '\n';

    usleep(100);

    ReadLM82(chip, &lm82_ctrl);
    cout << " ack lm82 data = " << lm82_ctrl << '\n';
    ReadLM82(chip, &lm82_ctrl);
    cout << " ack lm82 data = " << lm82_ctrl << '\n';

    return 0;

    // address
    /// cout << "Address : " << '\n';
  for (int i=6; i>=0; i--) {
    ///cout << "i = " << i << '\n';

    SetLM82(chip, lm82_ctrl &= ~0x1); // lower clock

    serial_data & 1<<i ? lm82_ctrl = 0x2 : lm82_ctrl = 0x0;
    SetLM82(chip, lm82_ctrl &= ~0x1); // lower clock
    SetLM82(chip, lm82_ctrl); // set data, with clock low
    SetLM82(chip, lm82_ctrl |= 0x1); // toggle clock
  }

  ///cout << "Read bit : " << '\n';

  // Set Read operation
    SetLM82(chip, lm82_ctrl &= ~0x1); // lower clock
  SetLM82(chip, lm82_ctrl = 0x2);
  SetLM82(chip, lm82_ctrl |= 0x1); // toggle clock

  /// cout << "Ack bit : " << '\n';

  // SDO high for Ack
    SetLM82(chip, lm82_ctrl &= ~0x1); // lower clock
    SetLM82(chip, lm82_ctrl |= 0x2); // sdo high to tristate
    SetLM82(chip, lm82_ctrl |= 0x1); // toggle clock

  // SDO low for No Ack
    //SetLM82(chip, lm82_ctrl = 0x0);
    //SetLM82(chip, lm82_ctrl |= 0x1); // toggle clock

    ReadLM82(chip, &lm82_ctrl);
    /// cout << " ack lm82 data = " << lm82_ctrl << '\n';

    ///cout << "Data readout : " << '\n';

  *data = 0;
  // Read out 8 data bits
  for (int i=7; i>=0; i--) {
    // toggle clock to extract next data bit
    SetLM82(chip, lm82_ctrl &= ~0x1);
  SetLM82(chip, lm82_ctrl |= 0x1);

  ReadLM82(chip, &lm82_ctrl);
    // readback data bit
    temp = lm82_ctrl >> 2;
    *data |= temp << i;
    ///cout << "i = " << i << " lm82_ctrl = " << lm82_ctrl << " data = " << *data << '\n';
  }
  /// cout << "No Ack : " << '\n';

  // SDO low for No Ack
  SetLM82(chip, lm82_ctrl = 0x0);
  SetLM82(chip, lm82_ctrl |= 0x1); // toggle clock

  /// cout << "Stop : " << '\n';

  // Stop : clock high, data low -> high
  SetLM82(chip, 0x1);
  SetLM82(chip, 0x3);

  return 0;
}

int FFv1Object::SetLM82(int chip, int data)
{
  //cout << "Chip # " << chip << " LM82 Set = $ " << hex << data << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x19, 3, 0);
  unsigned int fed_data = data<<(32-3);
  /// cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';

  // return 0;
  // usleep(10);

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data, 0x1);

  ffv1SingleWrite(2, 0x200);  // send cmd string

  return 0;
}

int FFv1Object::ReadLM82(int chip, unsigned int* data)
{
  /// cout << "Readback Chip # " << chip << " LM82" << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x19, 3, 1);
  /// cout << "fed_cmd = $ " << hex << fed_cmd << '\n';

  // return 0;
  // usleep(100);

  ffv1SingleWrite(fed_cmd, 0x201);  // sends read cmd
  // usleep(100);
  unsigned long fed_data = *data;
  ffv1SingleRead(&fed_data, 0x0);  // readback memory

  //cout << "fed_data = $ " << hex << fed_data << '\n';
  *data = fed_data >> (32-3);

  return 0;
}

int FFv1Object::SetDAC(int chip, int chan, int data)
{
    cout << "FE Chip # " << dec << chip << " DAC chan = " << chan << " Set DAC value(dec) " << data << '\n';

  unsigned int serial_data = (chan-1) << 8 | data;
  unsigned int vref_ctrl = 0;

  //  cout << "Serial data = 0x " << hex << serial_data << dec << '\n';

  SetAD8802(chip, 0x11); // clock low
  SetAD8802(chip, 0x1c); // enable CS#

  for (int i=11; i>=0; i--) {
        cout << "i = " << i << '\n';
    serial_data & 1<<i ? vref_ctrl = 0x1c : vref_ctrl = 0x18;
    SetAD8802(chip, vref_ctrl); // set data
    SetAD8802(chip, vref_ctrl | 0x2); // clock low to high
  }

  SetAD8802(chip, 0x1f); // disable CS# (clock should be high when done)

  return 0;
}

int FFv1Object::SetAD8802(int chip, int data)
{
  cout << "FE Chip # " << chip << " AD8802 = $ " << hex << data << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x18, 5, 0);
  unsigned int fed_data = data<<(32-5);
  cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';

  // return 0;

  // usleep(10);

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data, 0x1);

  ffv1SingleWrite(2, 0x200);  // send cmd string

  return 0;
}

int FFv1Object::ReadAD8802(int chip, unsigned int* data)
{
  cout << "Readback Chip # " << chip << " AD8802" << '\n';
  unsigned int fed_cmd = fedCmd(chip, 0x18, 5, 1);
  cout << "fed_cmd = $ " << hex << fed_cmd << '\n';

  // return 0;

  ffv1SingleWrite(fed_cmd, 0x201);  // sends read cmd
  // usleep(10);
  unsigned long fed_data = *data;
  ffv1SingleRead(&fed_data, 0x0);  // readback memory
  *data = fed_data >> (32-5);

  return 0;
}


unsigned int FFv1Object::fedCmd(int chip_id, int cmd_desig, int cmd_len, int r_w)
{

  unsigned int cmd_word = 0;
  //  unsigned int swap_word = 0;


#define chip_id_mask 0xe0ffffff
#define cmd_desig_mask 0xffc1ffff
#define cmd_len_mask 0xffff0000

  //  cout << "chip id = " << chip_id << " cmd designator = $ " << hex << cmd_desig << " cmd length = " << dec << cmd_len << " R/W = " << r_w << '\n';

  // construct the 32 bit cmd string

  cmd_word = 0x20810000;

/*    x = (chip_id << 24) | cmd_word; */
/*    printf(" x = 0x%08x \n", x ); */

  if (r_w == 1) cmd_len++; // read command length must be one more than write
  else if (r_w == 2)
    {
      cmd_len+=2; // read only regs need 2 extra bit shifts
      r_w = 1; // must put back normal read bit to make cmd_word
    }

  cmd_word |= r_w << 22 | chip_id << 24 | cmd_desig << 17 | cmd_len;

  // cout << "fed cmd = $ " << hex << cmd_word << '\n';

  return cmd_word;

  // no more swapping needed.

//    // finally swap bits around as hardware expects msb's first.. uggh
//    // i.e. swap pairs of bits 31:0 ; 30:1 ; 29:2 ... 16:15

//    for (i=0; i<16; i++) {
//      swap_word |= (cmd_word >> (31-2*i) & 1 << i) | (cmd_word << (31-2*i)) & 1 << (31-i);
//    }

//    printf("bit swapped command = 0x%08x \n", swap_word);

  return 0;
}

unsigned int FFv1Object::cdcCmd(int chip_id, int cmd_desig, int cmd_len)
{

  unsigned int cmd_word = 0;

  cmd_word = 0x90200000;

  cmd_word |= chip_id << 29 | cmd_desig << 22 | cmd_len << 5;

  cout << "cdc cmd = $ " << hex << cmd_word << '\n';

  return cmd_word;
}

int FFv1Object::FillVMEMem(int data)
{
  cout << " Fill FED VME Memory buffer with = 0x" << hex << data << dec <<'\n';

  // usleep(500);

  // 2 KB BRAM => 512 long words
  for (int i=0; i<512; i++) {
  ffv1SingleWrite(data, i);
  }

  return 0;
}

int FFv1Object::FillVMEEventMem(int data)
{
  // cout << " Fill FED VME Event buffer with = 0x" << hex << data << dec <<'\n';

  // 32 KB event memory
   for (int i=0; i<32*1024/4; i++) {
     ffv1SingleWrite(data, FED_Readout_BRAM/4 + i);
   }
  return 0;
}

int FFv1Object::DumpVMEMem(void)
{
  unsigned long int data;

  cout << " FED VME Memory Buffer:" <<'\n';

  // usleep(500);

  // show first few locations
  for (int i=0; i<8; i++) {
  ffv1SingleRead(&data, i);
  cout << hex <<  data << " , ";
  }
  cout << '\n';

  return 0;
}

int FFv1Object::SetClockSkew(int fe_chip, int delay_chip, int delay_chan, int coarse, int fine)
{
//    // CDC dial thru command

  cout << dec << "FE Chip # " << fe_chip << " Delay chip # " << delay_chip << " Coarse = "  << coarse << " Fine = " << fine << '\n';

  unsigned int cdc_len = 36;  // changed back from 37 to 36 on 23.09.2003
  //  unsigned int tot_len = cdc_len + 27;
  unsigned int tot_len = 63;  // changed from 64 to 63 on 23.09.2003

//    printf("enter cdc cmd length => " );
//    cin >> cdc_len;
//    printf("enter total cmd length => " );
//    cin >> tot_len;

  printf("total cmd len = %d; cdc len = %d \n", tot_len, cdc_len );
  printf("clearing memory location after command string! \n" );

  unsigned int fed_cmd = fedCmd(fe_chip, 0x15, tot_len, 0);
  unsigned int cdc_cmd = cdcCmd(delay_chip, 0x01, cdc_len);

  unsigned int fed_data1 = cdc_cmd | coarse << 1 | fine >> 4;
  unsigned int fed_data2 = fine << 28 | coarse << 24 | fine << 19 | coarse << 15 | fine << 10 | coarse << 6 | fine << 1;
  cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data1 = $ " << fed_data1 << " ; fed_data2 = $ " << fed_data2 << '\n';

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data1, 0x1); // cdc cmd and data
  ffv1SingleWrite(fed_data2, 0x2);

  ffv1SingleWrite(0, 0x3);  // 26.09.03 ; clear next word to avoid hangup in serial decoder (this stops every second command being ignored!)

  ffv1SingleWrite(3, 0x200);  // send cmd string

  return 0;
}

 // nb len = 33 because no code 0
int be_cmd_len[33] = {0, 3, 2, 16, 32, 8, 2, 2, 1, 1, 12, 1, 17, 0, 0, 0, 32, 24, 12, 24, 1, 32, 18, 24, 16, 16, 0, 0, 0, 0, 0, 0, 0};

const char* be_cmd_name[33] = { "", "Trigger Select", "Mode", "TTCrx I2C Wr", "Test Register", "FE Enable", "BE Readout", "BE Run", "Software Trigger", "Soft Reset", "FED ID", "TTCrx Reset", "LM82 I2C Ctrl", "", "", "", "BE Status", "L1A Ctr", "BX Ctr", "Unread Frame Ctr", "TTC Ready", "Firmware ID", "Unread Word Ctr", "Total Frame Ctr", "TTCrx I2C Rd", "LM82 I2C Rd", "", "", "", "", "", "", "" };

int FFv1Object::BECommand(int be_code, int be_rw, unsigned int* be_data, int verbose)
{
  unsigned int fed_cmd;
  unsigned long fed_data, fed_data2;

  if (verbose > 1) cout << dec << "BE Chip cmd code # " << be_code << " R/W = " << be_rw << '\n';

  if (be_rw == 0)
    {
      fed_cmd = fedCmd(10, be_code, be_cmd_len[be_code], 0);
      fed_data = *be_data<<(32-be_cmd_len[be_code]);

      if (verbose > 1) cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';
      if (verbose > 1) cout << "BE command name = " << be_cmd_name[be_code] << '\n';

      ffv1SingleWrite(fed_cmd, FED_Serial_BRAM/4);  // fill cmd string in buffer
      ffv1SingleWrite(fed_data, (FED_Serial_BRAM+4)/4);

      ffv1SingleWrite(2, FED_Serial_Write/4);  // send cmd string
    }
  else
  {
    if (be_code >= 16) // readonly regs
      {
	fed_cmd = fedCmd(10, be_code, be_cmd_len[be_code], 2); // r_w =2 must add 2 to length command
      }
    else
      {
	fed_cmd = fedCmd(10, be_code, be_cmd_len[be_code], 1);
      }

    ffv1SingleWrite(fed_cmd, FED_Serial_Read/4);  // sends read cmd
          usleep(100);
    // should check Ser Status register here
    ffv1SingleRead(&fed_data, FED_Serial_BRAM/4);  // readback memory

    if (be_code >=16 ) // readonly registers, must compensate for an extra bit shifted out by serial logic
      {
	fed_data <<= 1 ;
	ffv1SingleRead(&fed_data2, (FED_Serial_BRAM+4)/4);  // in case of 32 bit values where last bit appears in next memory location
	fed_data |= (fed_data2 >> 31);
      }

    *be_data = fed_data>>(32-be_cmd_len[be_code]);

    if (verbose > 1) cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';
    if (verbose > 1) cout << "BE command name = " << be_cmd_name[be_code] << '\n';
    if (verbose > 1) cout << "Reading: data (hex) = " << hex << *be_data << '\n';
  }
 
  if (verbose <= 1)
    {    
       usleep(1000); // need to wait between serial commands
    }

  return 0;
}

int FFv1Object::DisplayBEStatus(int flag)
{
  unsigned int l1a_cnt, bx_cnt, unread_frame_cnt, unread_word_cnt, total_frame_cnt;
  unsigned int be_status, ttc_status, ttc_ready;
  unsigned int fpga_code;
  unsigned int verbose = 0;

  if (flag == 1)
    {
      unsigned int trig_src, be_mode, ttc_ctrl, test_reg, fe_enable, be_link, be_enable;
      unsigned int fed_id;

      BECommand(1, 1, &trig_src, verbose);
      BECommand(2, 1, &be_mode, verbose);
      BECommand(3, 1, &ttc_ctrl, verbose);
      BECommand(4, 1, &test_reg, verbose);
      BECommand(5, 1, &fe_enable, verbose);
      BECommand(6, 1, &be_link, verbose);
      BECommand(7, 1, &be_enable, verbose);
      BECommand(10, 1, &fed_id, verbose);

      cout << "BE Control Registers: " << '\n';
      cout << "1) Trigger Source = $ " << hex << trig_src << '\n';
      cout << "2)        BE Mode = $ " << hex << be_mode << '\n';
      cout << "3)     TTCrx Ctrl = $ " << hex << ttc_ctrl << '\n';
      cout << "4)  Test Register = $ " << hex << test_reg << '\n';
      cout << "5)      FE Enable = $ " << hex << fe_enable << '\n';
      cout << "6)        BE Link = $ " << hex << be_link << '\n';
      cout << "7)      BE Enable = $ " << hex << be_enable << '\n';

      cout << "10)    FED ID Reg = $ " << hex << fed_id << '\n';
      cout << '\n';
    }


  // BECommand(24, 1, &ttc_status, verbose);

  BECommand(16, 1, &be_status, verbose);

  BECommand(17, 1, &l1a_cnt, verbose);
  BECommand(18, 1, &bx_cnt, verbose);
  BECommand(19, 1, &unread_frame_cnt, verbose);
  BECommand(20, 1, &ttc_ready, verbose);
  BECommand(21, 1, &fpga_code, verbose);
  BECommand(22, 1, &unread_word_cnt, verbose);
  BECommand(23, 1, &total_frame_cnt, verbose);

  cout << "BE Status Registers: " << '\n';
  cout << "21)    FPGA Usercode = $ " << hex << fpga_code << '\n';
  cout << "16)        BE Status = $ " << hex << be_status << '\n';
  cout << "17)          L1A ctr = " << dec << l1a_cnt << '\n';
  cout << "18)           BX ctr = " << dec << bx_cnt << '\n';
  cout << "23)  Total Frame ctr = " << dec << total_frame_cnt << '\n';
  cout << "19) Unread Frame ctr = " << dec << unread_frame_cnt << '\n';
  cout << "22) Unread Word (64) ctr = " << dec << unread_word_cnt << " $ " << hex << unread_word_cnt << '\n';
  //  cout << "24)     TTC I2C Read = $ " << hex << ttc_status << '\n';
  cout << "20)        TTC Ready = $ " << hex << ttc_ready << '\n';

  return 0;

}

int FFv1Object::InitBE(int run_type, unsigned int trigger_source)
{
  unsigned int  be_mode, ttc_ctrl, test_reg, fe_enable, be_link, be_enable;
  unsigned int fed_id;
  unsigned int dummy = 1;
  unsigned int verbose = 0;

  test_reg = 0x00; // disable test patterns, disable ignoring of vme throttle
  fe_enable = 0xff; // all enabled
  //fe_enable = 0x00; // all disabled
  be_link = 0x02; // v-link
  be_enable = 0x03; // accept triggers, accept frames

  ttc_ctrl = 0x0; // to avoid compilation warning messages

  fed_id = 0xfed;

  if (run_type == 1)
    {
      be_mode = 0x01; // scope
    }
  else if (run_type == 2 || run_type == 3)
    {
      be_mode = 0x02; // ff
    }
  else
    {
      cout << "Sorry Illegal Run Type " << '\n';
      return 1;
    }

  BECommand(9, 0, &dummy, verbose); // reset status/ counters, value written not important

  BECommand(1, 0, &trigger_source, verbose);

  BECommand(2, 0, &be_mode, verbose);
  BECommand(4, 0, &test_reg, verbose);
  BECommand(5, 0, &fe_enable, verbose);
  BECommand(6, 0, &be_link, verbose);

  BECommand(10, 0, &fed_id, verbose);

  BECommand(7, 0, &be_enable, verbose); // should be last command ?

  return 0;

}



int FFv1Object::TestFrameFinding(int num_triggers, int num_microsecs)
{
  // simulate a frame finding run

  cout << "Simulate Frame Finding with: Nr Triggers = " << dec << num_triggers << " and apv frames for " << num_microsecs << " microsecs" << '\n';

  unsigned int test_reg, saved;
  unsigned int verbose = 0; // 2 disables usleep() inside BECommand


  // send dummy triggers to fill l1a/bx fifos
  for (int i = 0; i<num_triggers; i++)
    {
      SendSWTrig();
    }

  BECommand(4, 1, &test_reg, verbose);
  //  usleep(1000);

  saved = test_reg;
  test_reg |= 0x1; // enable apv frame generator

  // enable apv frames for a bit
  BECommand(4, 0, &test_reg, 2); // start apv frame patterns

  usleep(num_microsecs);

  BECommand(4, 0, &saved, verbose); // stop patterns

  return 0;
}

 // nb len = 33 because no code 0
int fe_cmd_len[33] = {0, 63, 24, 32, 60, 17, 14, 1, 40880, 0, 240, 1, 4, 36, 12, 192, 5, 10, 0, 0, 145, 0, 8, 24, 0, 0, 1, 16, 0, 0, 0, 0, 0};

const char* fe_cmd_name[33] = { "", "Load Skew", "Load Tick", "Firmware ID", "Load Thresh", "LM82 I2C Wr", "TrimDAC Ctrl", "Spy Arm", "Spy Fire", "", "Load Median", "Median Enable", "Soft Reset", "Load Ped Data", "Set Ped Addr", "Valid Strips", "Mode", "Scope Length", "", "", "Monitor", "CDC Thru", "OptoRx Ctrl", "ADC Ctrl", "Unused", "Unused", "Load Complement", "LM82 I2C Rd", "", "", "", "", "" };

int FFv1Object::FECommand(int fe_id, int fe_code, int fe_rw, unsigned int* fe_data, int verbose)
{
  unsigned int fed_cmd;
  unsigned long fed_data, fed_data2;

  if (verbose > 1) cout << dec << "FE Chip # " << fe_id << "cmd code # " << fe_code << " R/W = " << fe_rw << '\n';

  if (fe_rw == 0)
    {
      fed_cmd = fedCmd(fe_id, fe_code, fe_cmd_len[fe_code], 0);
      fed_data = *fe_data<<(32-fe_cmd_len[fe_code]);

      if (verbose > 1) cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';
      if (verbose > 1) cout << "FE command name = " << fe_cmd_name[fe_code] << '\n';

      ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
      ffv1SingleWrite(fed_data, 0x1);

      ffv1SingleWrite(2, 0x200);  // send cmd string
    }
  else
  {
    if (fe_code == 3 || fe_code == 20 || fe_code == 27) // readonly regs
      {
	fed_cmd = fedCmd(fe_id, fe_code, fe_cmd_len[fe_code], 2); // r_w =2 must add 2 to length command
      }
    else
      {
	fed_cmd = fedCmd(fe_id, fe_code, fe_cmd_len[fe_code], 1);
      }

    ffv1SingleWrite(fed_cmd, 0x201);  // sends read cmd
    usleep(100);
    ffv1SingleRead(&fed_data, 0x0);  // readback memory

    if (fe_code == 3 || fe_code == 20 || fe_code == 27) // readonly regs
    // readonly registers, must compensate for an extra bit shifted out by serial logic
      {
	fed_data <<= 1 ;
	ffv1SingleRead(&fed_data2, 0x1);  // in case of 32 bit values where last bit appears in next memory location
	fed_data |= (fed_data2 >> 31);
      }

    *fe_data = fed_data>>(32-fe_cmd_len[fe_code]);

    if (verbose > 1) cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';
    if (verbose > 1) cout << "FE command name = " << fe_cmd_name[fe_code] << '\n';
    if (verbose > 1) cout << "Reading: data (hex) = " << hex << *fe_data << '\n';
  }
 
  if (verbose <= 1)
    {    
      usleep(1000); // need to wait between serial commands
    }

  return 0;
}

int FFv1Object::ClearReadoutCSR(int verbose)
{
  // clearing csr allows next event to be readout
     unsigned long readout_status;
     unsigned int clear = 0;

      ffv1SingleRead(&readout_status, FED_Readout_CSR/4);
      if (verbose > 1) cout << "Before  Clear Event Pending: Status = " << readout_status << '\n';
      usleep(100);

      ffv1SingleWrite(clear, FED_Readout_CSR/4);

      usleep(1000);
      ffv1SingleRead(&readout_status, FED_Readout_CSR/4);
      if (verbose > 1) cout << "After Clear Event Pending: Status = " << readout_status << '\n';

      return 0;
}

int FFv1Object::ResetFED(int verbose)
{
  // reset register in VME resets all FPGAs and clock chain
     unsigned int clear = 0;

      ffv1SingleWrite(clear, FED_Reset/4);
      usleep(100);

      if (verbose > 1) cout << "Sent a RESET to the FED" << '\n';
  
      usleep(2000000);  // safe wait for DCMs to lock

      if (verbose > 1) cout << "Clear VME Event Memory" << '\n';
      FillVMEEventMem(0xbeefface);

      return 0;
}

int FFv1Object::TestReadout(int verbose)
{
  unsigned long readout_status;
  unsigned long buffer_length;
  unsigned long event_length;
  unsigned long event_number;

  unsigned long data;

  // poll on vme readout status

  unsigned int num_secs = 6;
  unsigned int num_triggers = 1;
  unsigned int dummy;
  unsigned int i = 0, j=0;

  unsigned int start = (unsigned int) time((time_t*) &dummy);
  unsigned int now = (unsigned int) time((time_t*) &dummy);

  while ((now - start) < num_secs) {

    if (j < num_triggers) {
      SendSWTrig();
      j++;
    }

    ffv1SingleRead(&readout_status, FED_Readout_CSR/4);
    ffv1SingleRead(&buffer_length, FED_Readout_Buffer_Length/4);
    ffv1SingleRead(&event_length, FED_Readout_Event_Length/4);
    ffv1SingleRead(&event_number, FED_Readout_Event_Ctr/4);

    if (readout_status & 0x1) {
      if (verbose > 1) cout << "Event Pending: Status = " << readout_status << '\n';
      if (verbose > 1) cout << "Event Number = " << event_number << '\n';
      if (verbose > 1) cout << "Buffer Length (32) = " << buffer_length << " $ " << hex << buffer_length << '\n';
      if (verbose > 1) cout << "Event Length (32) = " << event_length << " $ " << hex << event_length << '\n';

      // readout event
      cout << "Sample |  Data " << '\n';

      for (i=0; i<16; i++) {

	ffv1SingleRead(&data, FED_Readout_BRAM/4 + i);

	// cout << showbase;
	cout << hex << setw(8) << i << "\t 0x" << data << '\n';
      }

      if (buffer_length < MAX_BUFFER_LENGTH) {

      unsigned long* bufp = local_buffer;
      for (i=0; i<buffer_length; i++) {
	ffv1SingleRead((long unsigned int*) bufp, FED_Readout_BRAM/4 + i);
	bufp++;
      }

      } else {
	cout << "Warning: Buffer Length too long. Skipping data readout" << '\n';
      }
	
//        for (i=0; i<8; i++) {
//  	cout << hex << setw(8) << i << "\t 0x" << local_buffer[i] << '\n';
//        }

      CheckEvent(local_buffer, buffer_length);

      // clear csr to allow next event in BRAM
      usleep(10);
      unsigned int clear = 0;
      ffv1SingleWrite(clear, FED_Readout_CSR/4);
      // usleep(100);
      ffv1SingleRead(&readout_status, FED_Readout_CSR/4);
      if (verbose > 1) cout << "After Clear Event Pending: Status = " << readout_status << '\n';
    }
    now = (unsigned int) time((time_t*) &dummy);

  }

  return 0;
}

int FFv1Object::CheckEvent(unsigned long* event_buffer, unsigned long event_length)
{
  unsigned int event_type;
  unsigned int l1a_nr;
  unsigned int bx_nr;
  unsigned int fed_id;
  unsigned int crc;
  unsigned int evt_length;

  unsigned int fe_pipeaddr[8];
  unsigned int fe_len[8];
  unsigned int fe_status[8][12];

  vector<unsigned char> fe_data[8];
  //unsigned int fe_data[12][1024];

  // daq header
  event_type = event_buffer[0] >> 24 & 0xf;
  l1a_nr = event_buffer[0] & 0xffffff;
  bx_nr = event_buffer[1] >> 20 & 0xfff;
  fed_id = event_buffer[1] >> 8 & 0xfff;

  if (fed_id != 0xFED) {
	cout << "Error: fed_id != 0xFED. Stopping event check." << '\n';
	return 1;
  }

  // tracker header
  for (int i=0; i<8; i++) {  // loop over fe fpgas

    fe_len[i] = event_buffer[i*4 + 4] >> 16 & 0xffff;
    fe_pipeaddr[i] = event_buffer[i*4 + 5] >> 8 & 0xff;
    fe_status[i][0] = event_buffer[i*4 + 5] >> 2 & 0x3f;
    fe_status[i][1] = ((event_buffer[i*4 + 2] & 0x3) << 4) | (event_buffer[i*4 + 4] >> 28 & 0xf);
    fe_status[i][2] = event_buffer[i*4 + 2] >> 22  & 0x3f;
    fe_status[i][3] = event_buffer[i*4 + 2] >> 16  & 0x3f;
    fe_status[i][4] = event_buffer[i*4 + 2] >> 10  & 0x3f;
    fe_status[i][5] = event_buffer[i*4 + 2] >> 4  & 0x3f;
    fe_status[i][6] = ((event_buffer[i*4 + 2] & 0xf) << 2) | (event_buffer[i*4 + 3] >> 30 & 0x3);
    fe_status[i][7] = event_buffer[i*4 + 3] >> 24  & 0x3f;
    fe_status[i][8] = event_buffer[i*4 + 3] >> 18  & 0x3f;
    fe_status[i][9] = event_buffer[i*4 + 3] >> 12  & 0x3f;
    fe_status[i][10] = event_buffer[i*4 + 3] >> 6  & 0x3f;
    fe_status[i][11] = event_buffer[i*4 + 3] & 0x3f;

  }

  printf("fe chan | length | pipe addr | apv status \n");
  for (int i=0; i<8; i++) {  // loop over fe fpgas
    printf("%d    $ %04x ( %d )    $ %04x ( %d ) \n", i, fe_len[i], fe_len[i], fe_pipeaddr[i], fe_pipeaddr[i]);
  }

   unsigned int nwords = 33; // skip headers
  // put tracker data  into a vector of bytes for later manipulations
  for (int i=0; i<8; i++) {  // loop over fe fpgas
    if (fe_len[i] != 0) {
      int k;
      unsigned char value;
      for (unsigned int nbytes=0; nbytes<fe_len[i]; nbytes++) {
	k = nbytes%4;
	if (k == 0) nwords++;
	value = (event_buffer[nwords] >> (24 - k*8)) & 0xff;
	//	printf("word # %d ; byte # %d ; k = %d ; buffer = $ %08x ; value = $ %02x \n",nwords, nbytes, k, event_buffer[nwords], value);
	fe_data[i].push_back(value);
      }
      nwords++; // skip padding at end of each FE
    }
  }

//    // test vector
//    for (int i=0; i<2; i++) {  // loop over fe fpgas
//      if (fe_len[i] != 0) {
//        unsigned int nwords = 14; // skip headers
//        for (unsigned char j=0; j<6; j++) {
//  	fe_data[i].push_back(j);
//  	nwords++;
//        }
//      }
//    }

  for (int i=0; i<1; i++) {  // loop over fe fpgas

      printf("\n+++++++++++++++++\nFE # %d \n", i);

    int m = 0;
    for (vector<unsigned char>::iterator it = fe_data[i].begin();
	 it != fe_data[i].end(); ++it) {
      m++;
            printf("byte # %d ; $ %02x \n",m, *it);

      if ( m == 1 ) {
	// fe_fibre_code[i*8 + m] = fe_data[i][m];
	// printf("code = ; $ %02x \n", fe_data[i][m]);
      }

      // cout << dec << *it << endl;
    }
  }

  evt_length = event_buffer[event_length - 1] & 0xffffff;
  crc = event_buffer[event_length - 2] >> 16 & 0xffff;


  return 0;
}


int FFv1Object::DisplayVMEStatus(int verbose)
{
  unsigned long firmware_id;
  unsigned long clock_select;
  unsigned long vme_status;

  unsigned long ttc_clk_ctr[2];
  unsigned long bp_clk_ctr[2];

  unsigned long readout_status;
  unsigned long buffer_length;
  unsigned long event_length;
  unsigned long event_number;

  unsigned long data, data1, data2;

 
   if (verbose > 1) cout << '\n' << "VME Registers: " << '\n';

   ffv1SingleRead(&firmware_id, FED_Firmware_ID/4);
   ffv1SingleRead(&clock_select, FED_Clock_Select/4);
   ffv1SingleRead(&vme_status, FED_VME_Status/4);

   if (verbose > 1) cout << "Firmware ID = $ " << hex << firmware_id << '\n';
   if (verbose > 1) cout << "VME Status Reg = $ " << hex << vme_status << '\n';
   if (verbose > 1) cout << "Clock Select = " << dec << clock_select << '\n';
    
   ffv1SingleRead(&ttc_clk_ctr[0], FED_TTC_Clk_Ctr/4);
   usleep(100);
   ffv1SingleRead(&ttc_clk_ctr[1], FED_TTC_Clk_Ctr/4);

   ffv1SingleRead(&bp_clk_ctr[0], FED_BP_Clk_Ctr/4);
   usleep(100);
   ffv1SingleRead(&bp_clk_ctr[1], FED_BP_Clk_Ctr/4);

   if (verbose > 1) cout << "TTC Clock Ctr = " << ttc_clk_ctr[0] << '\n';
   if (verbose > 1) cout << "BP Clock Ctr = " << bp_clk_ctr[0] << '\n';

   if (verbose > 1) cout << "TTC Clock Ctr (after 100 microsec) = " << ttc_clk_ctr[1] << '\n';
   if (verbose > 1) cout << "BP Clock Ctr (after 100 microsec)  = " << bp_clk_ctr[1] << '\n';

   if (verbose > 1) cout << "TTC Clock Ctr change = " << ttc_clk_ctr[1] - ttc_clk_ctr[0] << '\n';
   if (verbose > 1) cout << "BP Clock Ctr change  = " << bp_clk_ctr[1]- bp_clk_ctr[0] << '\n';

   ffv1SingleRead(&readout_status, FED_Readout_CSR/4);
   ffv1SingleRead(&buffer_length, FED_Readout_Buffer_Length/4);
   ffv1SingleRead(&event_length, FED_Readout_Event_Length/4);
   ffv1SingleRead(&event_number, FED_Readout_Event_Ctr/4);

   if (verbose > 1) cout << '\n' << "Readout Status: " << '\n';
   if (verbose > 1) cout << "Event Pending: Status = " << readout_status << '\n';
   if (verbose > 1) cout << "Event Number = " << event_number << '\n';
   if (verbose > 1) cout << "Buffer Length (32) = " << dec << buffer_length << " $ " << hex << buffer_length << '\n';
   if (verbose > 1) cout << "Event Length (32) = " << dec << event_length << " $ " << hex << event_length << '\n';

   // readout event
   //   cout << "Sample\t\t |  Data " << '\n';

//     for (int i=0; i<16; i++) {

//       ffv1SingleRead(&data, FED_Readout_BRAM/4 + i);

//       // cout << showbase;
//       // cout << hex << setw(8) << setfill(0) << i << "\t 0x" << data << '\n';
//       cout.width(8);
//       cout << hex << i << "\t 0x" << data << '\n';
//     }

   printf("Event Data => \n ");
   printf("(64 bit) word nr |  data (hex) \n");
   for (int i=0; i<1600; i+=2) {

     ffv1SingleRead(&data1, FED_Readout_BRAM/4 + i);
     ffv1SingleRead(&data2, FED_Readout_BRAM/4 + i+1);

     printf("           %03d    |   %08x%08x   \n", i/2, data1, data2);
   }

  return 0;
}

int FFv1Object::DisplayClockCounters(int verbose)
{
  unsigned long ttc_clk_ctr[2];
  unsigned long bp_clk_ctr[2];
  unsigned int ttc_ready;
 
  //  BECommand(20, 1, &ttc_ready, verbose);
  //  cout << "20)        TTC Ready = $ " << hex << ttc_ready << '\n';
   
   ffv1SingleRead(&ttc_clk_ctr[0], FED_TTC_Clk_Ctr/4);
   usleep(100);
   ffv1SingleRead(&ttc_clk_ctr[1], FED_TTC_Clk_Ctr/4);

   ffv1SingleRead(&bp_clk_ctr[0], FED_BP_Clk_Ctr/4);
   usleep(100);
   ffv1SingleRead(&bp_clk_ctr[1], FED_BP_Clk_Ctr/4);

   if (verbose > 1) cout << "TTC Clock Ctr = " << dec << ttc_clk_ctr[0] << '\n';
   if (verbose > 1) cout << "TTC Clock Ctr (after 100 microsec) = " << dec  << ttc_clk_ctr[1] << '\n';
   if (verbose > 1) cout << "TTC Clock Ctr change = " << dec << ttc_clk_ctr[1] - ttc_clk_ctr[0] << '\n';

   if (verbose > 1) cout << "BP Clock Ctr = " << dec << bp_clk_ctr[0] << '\n';
   if (verbose > 1) cout << "BP Clock Ctr (after 100 microsec)  = " << dec << bp_clk_ctr[1] << '\n';
   if (verbose > 1) cout << "BP Clock Ctr change  = " << dec << bp_clk_ctr[1]- bp_clk_ctr[0] << '\n';

   int error = 0;
   int max = 100;
   for (int i=0; i<max; i++) {
     BECommand(20, 1, &ttc_ready, verbose);
     if (ttc_ready != 1 ) error++;
   }
   cout << "Tested TTC Ready " << dec << max << " times; and found " << error << " errors " << '\n';

  return 0;
}

int FFv1Object::SelectClock(int clock, int verbose)
{
  unsigned long clock_select;
  unsigned long vme_status;

  unsigned long ttc_clk_ctr;
  unsigned long bp_clk_ctr;

 
  //   if (verbose > 1) cout << "Testing Clock Selection: " << '\n';

   ffv1SingleRead(&clock_select, FED_Clock_Select/4);
   //   ffv1SingleRead(&vme_status, FED_VME_Status/4);
   if (verbose > 1) cout << "Clock Select before change = " << dec << clock_select << '\n';
   //   if (verbose > 1) cout << "VME Status Reg = $ " << hex << vme_status << '\n';

//     ffv1SingleRead(&ttc_clk_ctr, FED_TTC_Clk_Ctr/4);
//     ffv1SingleRead(&bp_clk_ctr, FED_BP_Clk_Ctr/4);

//     if (verbose > 1) cout << "TTC Clock Ctr = " << ttc_clk_ctr << '\n';
//     if (verbose > 1) cout << "BP Clock Ctr = " << bp_clk_ctr << '\n';
//     usleep(100);

//     ffv1SingleRead(&ttc_clk_ctr, FED_TTC_Clk_Ctr/4);
//     ffv1SingleRead(&bp_clk_ctr, FED_BP_Clk_Ctr/4);

//     if (verbose > 1) cout << "TTC Clock Ctr = " << ttc_clk_ctr << '\n';
//     if (verbose > 1) cout << "BP Clock Ctr = " << bp_clk_ctr << '\n';

   if (verbose > 1) cout << "Trying to change clock setting to : " << clock << '\n';
   ffv1SingleWrite(clock, FED_Clock_Select/4);

// after changing clock do a vme reset
   ResetFED(2);

   ffv1SingleRead(&clock_select, FED_Clock_Select/4);
   ffv1SingleRead(&vme_status, FED_VME_Status/4);
   if (verbose > 1) cout << "Clock Selected = " << dec << clock_select << '\n';
   //   if (verbose > 1) cout << "VME Status Reg = $ " << hex << vme_status << '\n';

  return 0;
}

int FFv1Object::TestTTCrx(int* data)
{
  cout << "Read TTCrx Chip # Status Reg" << '\n';


  return 0;
}

int FFv1Object::SetTTCrx(int data)
{
  unsigned int fed_cmd = fedCmd(10, 0x3, 2, 0);
  unsigned int fed_data = data<<(32-2);
  /// cout << "fed_cmd = $ " << hex << fed_cmd << " ; fed_data = $ " << fed_data << '\n';

  // return 0;
  // usleep(10);

  ffv1SingleWrite(fed_cmd, 0x0);  // fill cmd string in buffer
  ffv1SingleWrite(fed_data, 0x1);

  ffv1SingleWrite(2, 0x200);  // send cmd string

  return 0;
}

int FFv1Object::ReadTTCrx(unsigned int* data)
{
  /// cout << "Readback Chip # " << chip << " LM82" << '\n';
  unsigned int fed_cmd = fedCmd(10, 0xb, 1, 1);
  /// cout << "fed_cmd = $ " << hex << fed_cmd << '\n';

  // return 0;
  // usleep(100);

  ffv1SingleWrite(fed_cmd, 0x201);  // sends read cmd
  // usleep(100);
  unsigned long fed_data = *data;
  ffv1SingleRead(&fed_data, 0x0);  // readback memory

  //cout << "fed_data = $ " << hex << fed_data << '\n';
  *data = fed_data >> (32-1);

  return 0;
}

#endif // _FFv1Object_hh_
